// ============================================================================
// ============================================================================
// ============================================================================
// ==                                                                        ==
// == Name    : TheEmuLib.Emu_Tile_XY.2.fsh                                  ==
// == Type    : Fragment shader                                              ==
// == Version : 1.0.0 (2017/02/03)                                           ==
// == Creator : TheEmu © TheEmu 2017, Some Rights Reserved                   ==
// == Licence : Creative Commons Attribution-ShareAlike 4.0                  ==
// ==           http://creativecommons.org/licences/by-sa/4.0                ==
// ==                                                                        ==
// == Purpose:  To fill a window with repetitions of an image with the tiles ==
// == being arranged on a rectangular grid.                                  ==
// ==                                                                        ==
// == Description: A single source image is used to tile the current window. ==
// == The basic tile grid is rectangular and parallel to the window's sides, ==
// == though the basic grid may be distorted by applying a rotation, a shear ==
// == or taper transform or any combination thereof.                         ==
// ==                                                                        ==
// == Alternate tiles,  horizontaly or vertically,  may optionaly be flipped ==
// == relative to their neighbours. The flips may be about a tile's vertical ==
// == or horizontal axes or about a diagonal axis, and may be combined.      ==
// ==                                                                        ==
// == Rows or columns of tiles may be shifted relative to their  neighbours. ==
// == The shifting may be applied only to alternate rows or  columns,  which ==
// == results in a zig-zag tile pattern, or progressive (proportional to the ==
// == row or column number) which produces a diagonal tile pattern.          ==
// ==                                                                        ==
// == This file is a member of The Emu's shader library.                     ==
// ==                                                                        ==
// == ====================================================================== ==
// ==                                                                        ==
// == Update history:                                                        ==
// ==                                                                        ==
// ==   2017/02/03 - v1.0.0 - Initial version, based on Emu_Tile_XY.1.fsh.   ==
// ==                                                                        ==
// ============================================================================
// ============================================================================
// ============================================================================

// ============================================================================
// == Standard shader inputs ==================================================
// ============================================================================

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// The image that is to be used as the tile.

uniform sampler2D iChannel0;

// ============================================================================
// == Imports from TheEmuLib ==================================================
// ============================================================================
//
// The GLSL shader language currently provides no mechanism for importing  any
// elements that are defined in other modules, not even C's crude source level
// #include mechanism. In the absence of anything better TheEmuLib handles any
// imports by manualy copying relevent utility code snippets from the  sources
// in the Shader Lib.Inc directory. This is very crude but I have attempted to
// be systematic in the way in which this is presented in the library sources.
//
// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Common_Utilities.lib.src

#define EMU_DEFAULT(type,x,default_value) ( (x==type(0.0)) ? (default_value) : (x) )

// = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

// Macros from TheEmuLib.Emu_Boolean_Vector_Ops.lib.src

#define Emu_bvec4_or(A,B)  bvec4 ( A[0]||B[0], A[1]||B[1], A[2]||B[2], A[3]||B[3] )
#define Emu_bvec4_and(A,B) bvec4 ( A[0]&&B[0], A[1]&&B[1], A[2]&&B[2], A[3]&&B[3] )
#define Emu_bvec4_xor(A,B) bvec4 ( A[0]^^B[0], A[1]^^B[1], A[2]^^B[2], A[3]^^B[3] )

// ============================================================================
// == Shader specific inputs ==================================================
// ============================================================================

// EmuLib standard scale and hotspot parameters.  Note, a hotspot of (0.5,0.5)
// may be required to produce a symetric result.

uniform vec2 Emu_Tile_XY_scale;
uniform vec2 Emu_Tile_XY_hotspot;

vec2 scale   = EMU_DEFAULT ( vec2, Emu_Tile_XY_scale,   vec2(1.0) );
vec2 hotspot = EMU_DEFAULT ( vec2, Emu_Tile_XY_hotspot, vec2(0.0) );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The shader's mode is controlled by a three decimal digit integer parameter.
// This encodes the following information:
//
//   Tile slip mode - controlled by the most significant digit
//   Tile size mode - controlled by the middle significant digit
//   Tile flip mode - controlled by the least significant digit
//
// The manner in which the tile size is specified is controlled by  the middle
// significant digit of the mode. This may be
//
//    0 - Use the default value of 1
//    1 - Emu_Tile_XY_size specifies the number of tile cells.
//    2 - Emu_Tile_XY_size specifies the absolute tile cell size.
//
// By default the tiles are used with no flips or being applied,  however this
// behaviour can be changed using the tile flip mode, This parameter comprises
// three bits which when set individualy encode the following options
//
//    Bit 0 - Flip tiles about their Y axes in alternate columns
//    Bit 1 - Flip tiles about their X axes in alternate rows
//    Bit 2 - Flip tiles about a diagonal in alternate rows and columns
//
// There are 8 possible flip patterns encoded as 0, 1, 2, 3, 4, 5, 6 and 7.
//
// By default no slips are applied,  but  this can be overriden using the tile
// slip mode and the separate tile slip parameter.  The slip mode is specified
// by the most significant digit of the mode and may be
//
//    0 - No tile slip is applied.
//    1 - Slip alternate rows.
//    2 - Slip alternate columns.
//    3 - Progressively slip all rows.
//    4 - Progressively slip all columns.
//
// Slip modes 1 and 2 produce zig-zag slip patterns, mode 3 and 4 diagonals.

uniform int Emu_Tile_XY_mode;
int base_mode = abs(Emu_Tile_XY_mode);

int slip_mode = EMU_DEFAULT ( int, int ( mod ( base_mode / 100.0, 10.0) ), 0 );
int size_mode = EMU_DEFAULT ( int, int ( mod ( base_mode /  10.0, 10.0) ), 1 );
int flip_mode = EMU_DEFAULT ( int, int ( mod ( base_mode /  1.00, 10.0) ), 0 );

bvec4 flip = bvec4 ( ( flip_mode & 1 ) != 0,
                     ( flip_mode & 2 ) != 0,
                     ( flip_mode & 4 ) != 0,
                     ( flip_mode & 4 ) != 0
                   );

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// The number of tile cells is specified by the size control parameter  which,
// depending on the size mode, either specifies the number of cells the window
// is to be divided into or the dimensions of the cells. The default for is to
// use a single cell.

uniform vec2 Emu_Tile_XY_size;

vec2 grid_x_02 = (size_mode==1) ? vec2(1.0) : u_WindowSize;

vec2 grid_size = EMU_DEFAULT ( vec2, Emu_Tile_XY_size, grid_x_02 );
vec2 grid_cell = (size_mode==2) ? grid_size : u_WindowSize / grid_size;

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the tiles are arranged on an orthogonal grid,  however, subject
// to the slip_mode (see above) the rows or columns may be  slipped  relative
// to each other with the amount of slippage being specified by the following
// parameter. The slippage is specified in units of the tile size.

uniform float Emu_Tile_XY_slip;
#define slip  Emu_Tile_XY_slip

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the origin of the tile coordinate system, which is defined by
// the hotspot parameter, will be at a tile corner. This can be modified by
// the following pffset parameter which specifies how much each tile is to
// shifted from the default position. It is in units of the tile size so a
// value of (0.5,0.5) moves the origin to a tile centre.

uniform vec2   Emu_Tile_XY_offset;
#define offset Emu_Tile_XY_offset

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

// By default the grid lines are parallel to the X and Y axes, however this
// can be modified by using the angle, skew and taper control parameters.
//
//    angle - rotates the grid by the specified angle in degrees
//    taper - tapers the grid by the specified factor
//    skew  - skews the grid by the specified factor
//
// A skew of (1.0,0.0) will leave the horizontal grid lines unchanged while
// causing the vertical grid lines to slope by 45 degrees. A positive taper
// causes the grid line spacing to narrow in the direction of the axis that
// the taper has been applied to. Skew and taper may be any value but it is
// expected that they will normaly be in the range -1.0 to +1.0.
//
// Note, the rotation is centered on the specified hotspot, see above. Also
// note,  that in order to obtain a symetric tapering a hotspot position of
// 0.5 in one or both directions needs to be specified.
//
// The defaults for all of these parameters are zero.

uniform vec2  Emu_Tile_XY_skew;
uniform vec2  Emu_Tile_XY_taper;
uniform float Emu_Tile_XY_angle;

#define skew  Emu_Tile_XY_skew
#define taper Emu_Tile_XY_taper

float angle = radians(Emu_Tile_XY_angle);

mat2 rotation = mat2 ( cos(angle), sin(angle), 
                      -sin(angle), cos(angle)
                     );

// ============================================================================
// == The shader's major function =============================================
// ============================================================================

vec4 Emu_Tile_1 ( vec2 uv0 )
 {
   vec2 uv = uv0;

   // Determine whether in an even or an odd column or row.

   bvec2 odd = notEqual ( floor(mod(uv,2.0)), vec2(0.0) );

   // Apply any tile slips that were specified.

   switch ( slip_mode )
    { case 1: uv.x -= slip * float(odd.y); break;
      case 2: uv.y -= slip * float(odd.x); break;
      case 3: uv.x -= slip * floor(uv.y);  break;
      case 4: uv.y -= slip * floor(uv.x);  break;
    }

   // Get the index into the un-flipped tile texture.

   vec2 st = fract ( uv );

   // Apply tile flips.

   flip = Emu_bvec4_and ( odd.xyxy, flip.xyzz );

   if ( flip.x ) st = vec2 ( 1.0-st.x,     st.y );
   if ( flip.y ) st = vec2 (     st.x, 1.0-st.y );
   if ( flip.z ) st = vec2 (     st.y, 1.0-st.x );
   if ( flip.w ) st = vec2 ( 1.0-st.y,     st.x );

   // Get the colour and opacity from the texture.

   return texture2D ( iChannel0, st );

 }

// ============================================================================
// == The shader's main routine ===============================================
// ============================================================================

void main ( void )
 { 
   // Get the coordinates of the current point.

   vec2 xy = ( gl_FragCoord.xy - hotspot*u_WindowSize ) / scale;

   // Apply any requested rotation, skew or taper.

   xy *= rotation;
   xy += xy.yx * skew;
   xy /= 1.0 - taper*(xy/u_WindowSize).yx;

   // Rescale to the tile grid cell size and recenter.

   xy = xy / grid_cell  -  offset;

vec2 edge = vec2(0.06);   ///////////// <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<

   // Get distances into the tile from each edge.

   vec2 d1 = fract ( xy ); // Distances from low edges.
   vec2 d2 = 1.0 - d1;     // Distances from high edges.

   // Convert to fractions of the edge thickness.

   vec2 f1 = clamp ( d1/edge, 0.0, 1.0 );
   vec2 f2 = clamp ( d2/edge, 0.0, 1.0 );

   // Determine which of the eight edge regions the pixel lies in.

   int region = 0;

   if ( d1.x < edge.x ) region += 1;
   if ( d2.x < edge.x ) region += 2;
   if ( d1.y < edge.y ) region += 4;
   if ( d2.y < edge.y ) region += 8;

   // Calculate the weights to apply.

   float[4] w1 = float[4] ( length ( vec2 ( 1.0-f1.x, 1.0-f1.y ) ),
                            length ( vec2 ( 1.0-f1.x,     f1.y ) ),
                            length ( vec2 (     f1.x, 1.0-f1.y ) ),
                            length ( vec2 (     f1.x,     f1.y ) )
                          );

   float[4] w2 = float[4] ( length ( vec2 ( 1.0-f2.x, 1.0-f2.y ) ),
                            length ( vec2 ( 1.0-f2.x,     f2.y ) ),
                            length ( vec2 (     f2.x, 1.0-f2.y ) ),
                            length ( vec2 (     f2.x,     f2.y ) )
                          );

   // Normalise the weights so they sum to 1.0.

   float ww = 1.0 / ( w1[0] + w1[1] + w1[2] + w1[3] );

   w1 = float[4] ( w1[0]*ww, w1[1]*ww, w1[2]*ww, w1[3]*ww );
   w2 = float[4] ( w2[0]*ww, w2[1]*ww, w2[2]*ww, w2[3]*ww );

   // Get the colours of each region that affct the current pixel
   // and then mix them using the previously calculated weights. 

   vec4 color;
   vec4[4] cc;

   switch ( region )
    { 
      case 0x00: color = Emu_Tile_1 ( xy );

                 break;

      case 0x01: cc[0] = Emu_Tile_1 ( xy );
                 cc[1] = Emu_Tile_1 ( xy - vec2(edge.x,0.0) );

                 color = mix ( cc[0], cc[1], vec4(0.5-f1.x*0.5) );

                 break;

      case 0x02: cc[0] = Emu_Tile_1 ( xy );
                 cc[1] = Emu_Tile_1 ( xy + vec2(edge.x,0.0) );

                 color = mix ( cc[0], cc[1], vec4(0.5-f2.x*0.5) );

                 break;

      case 0x04: cc[0] = Emu_Tile_1 ( xy );
                 cc[1] = Emu_Tile_1 ( xy - vec2(0.0,edge.y ));

                 color = mix ( cc[0], cc[1], vec4(0.5-f1.y*0.5) );

                 break;

      case 0x08: cc[0] = Emu_Tile_1 ( xy );
                 cc[1] = Emu_Tile_1 ( xy + vec2(0.0,edge.y) );

                 color = mix ( cc[0], cc[1], vec4(0.5-f2.y*0.5) );

                 break;

      case 0x05: cc[0] = Emu_Tile_1 ( xy );
                 cc[1] = Emu_Tile_1 ( xy - vec2 ( edge.x,   0.0  ) );
                 cc[2] = Emu_Tile_1 ( xy - vec2 (   0.0,  edge.y ) );
                 cc[3] = Emu_Tile_1 ( xy - vec2 ( edge.x, edge.y ) );

                 color = cc[0]*w1[0] + cc[1]*w1[1] + cc[2]*w1[2] + cc[3]*w1[3];

                 break;


      case 0x06: cc[0] = Emu_Tile_1 ( xy );
                 cc[1] = Emu_Tile_1 ( xy - vec2 ( edge.x,   0.0  ) );
                 cc[2] = Emu_Tile_1 ( xy - vec2 (   0.0,  edge.y ) );
                 cc[3] = Emu_Tile_1 ( xy - vec2 ( edge.x, edge.y ) );

                 color = cc[0]*w1[0] + cc[1]*w1[1] + cc[2]*w1[2] + cc[3]*w1[3];

                 break;

      case 0x09: cc[0] = Emu_Tile_1 ( xy );
                 cc[1] = Emu_Tile_1 ( xy - vec2 ( edge.x,   0.0  ) );
                 cc[2] = Emu_Tile_1 ( xy - vec2 (   0.0,  edge.y ) );
                 cc[3] = Emu_Tile_1 ( xy - vec2 ( edge.x, edge.y ) );

                 color = cc[0]*w1[0] + cc[1]*w1[1] + cc[2]*w1[2] + cc[3]*w1[3];

                 break;

      case 0x0a: cc[0] = Emu_Tile_1 ( xy );
                 cc[1] = Emu_Tile_1 ( xy - vec2 ( edge.x,   0.0  ) );
                 cc[2] = Emu_Tile_1 ( xy - vec2 (   0.0,  edge.y ) );
                 cc[3] = Emu_Tile_1 ( xy - vec2 ( edge.x, edge.y ) );

                 color = cc[0]*w1[0] + cc[1]*w1[1] + cc[2]*w1[2] + cc[3]*w1[3];

                 break;

    }

   // Set the colour and opacity of the current pixel.

   gl_FragColor = color * gl_Color;

}
